/***********************************************************************
 * Name: uarthandler.c
 *
 * Project: LPC43xx / LPC18xx example
 *
 * Description: Basic UART initialisation and TX functions
 *
 ***********************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/
                                                                                 

#include "uarthandler.h"
#include "type.h"

#ifdef LPC43XX
    #include "LPC43xx_TC2.H"
    #include "scu_LPC43xx.h"
#endif

#ifdef LPC18XX
    #include "LPC18xx.H"
    #include "scu_LPC18xx.h"
#endif



/*****************************************************************************
** Function name:       UART1Init
**
** Descriptions:        Initialize UART 1port, setup pin select,
**                      clock, parity, stop bits, FIFO, etc.
**
** parameters:          UART baudrate
**
** Returned value:      TRUE or FALSE (for later implementation)
**
*****************************************************************************/

uint32_t UART1Init( uint32_t baudrate )
{

    Status errorStatus = ERROR;

    #define UART_TER_TXEN           ((uint8_t)(1<<7))


#ifndef DLL_DLM
    uint32_t uClk;
    uint32_t DivAddVal, MulVal, best_DivAddVal, best_MulVal, tmp;
    uint64_t best_divisor, divisor;
    uint32_t current_error, best_error;
    uint32_t recalcbaud;
#endif


    // Select the correct UART RX/TX function for the LPC4350/LPC1850
    scu_pinmux(0xC,13,MD_PDN,FUNC2);    // PC.13, U1_TXD
    scu_pinmux(0xC,14,MD_PDN,FUNC2);    // PC.14, U1_RXD

#ifdef DLL_DLM
    // This is the "simple" version for different frequencies/baudrates
    // If the processor frequency or the baudrate is different, the settings here need to be changed
    // 120MHz: DLM = 0 , DLL = 6 ,  FDR = 0xE5  921600 baud
    // 120MHz: DLM = 0 , DLL = 47 , FDR = 0xD5  115200 baud
    //  72MHz: DLM = 0 , DLL = 4 ,  FDR = 0x51  921600 baud
    //  72MHz: DLM = 0 , DLL = 23 , FDR = 0xA7  115200 baud
    
    // Settings for 120MHz, 115200 baud
    LPC_UART1->LCR = 0x83;                          // 8 bits, no Parity, 1 Stop bit
    LPC_UART1->DLM = 0x00;
    LPC_UART1->DLL = 47;
    LPC_UART1->FDR = 0xD5;

    LPC_UART1->LCR = 0x03;                          // DLAB = 0
    LPC_UART1->FCR = 0x07;                          // Enable and reset TX and RX FIFO
    LPC_UART1->TER |= UART_TER_TXEN;                // TX is enabled by default after a reset
    LPC_UART1->IER = IER_RBR;                       // Enable UART1 RX interrupt 
    errorStatus = SUCCESS;
#else
    // Compared to a fixed setting this algorithm adds 536 bytes ROM code
    uClk = UART1Frequency;

    // In the Uart IP block, baud rate is calculated using FDR and DLL-DLM registers the formula is :
    // BaudRate= uClk * (MulVal/(MulVal+DivAddVal) / (16 * (DLL_DLM)
    // It involves floating point calculations. That's the reason the formulae are adjusted with multiply and divide method.
    // The value of mulFracDiv and dividerAddFracDiv should comply to the following conditions:
    // 0 < mulFracDiv <= 15, 0 <= dividerAddFracDiv <= 14 
    best_error = 0xFFFFFFFF; /* Worst case */
    best_DivAddVal = 0;
    best_MulVal = 0;
    best_divisor = 0;
    for (MulVal = 1 ; MulVal <= 15 ; MulVal++)
    {
        for (DivAddVal = 0 ; DivAddVal < MulVal ; DivAddVal++)
        {
          divisor = ((uint64_t)uClk<<28)*MulVal/(baudrate*(MulVal+DivAddVal));
          current_error = divisor & 0xFFFFFFFF;
          
          tmp = divisor>>32;
          
          /* Adjust error */
          if(current_error > (1<<31)){
            current_error = -current_error;
            tmp++;
            }
            
          if(tmp<1 || tmp>65536) /* Out of range */
          continue;
            
          if( current_error < best_error){
            best_error = current_error;
            best_divisor = tmp;
            best_DivAddVal = DivAddVal;
            best_MulVal = MulVal;
            if(best_error == 0) break;
            }
        } /* end of inner for loop */
        
        if (best_error == 0)
          break;
    } /* end of outer for loop  */
    
    if(best_divisor == 0) return ERROR; /* can not find best match */
    
    recalcbaud = (uClk>>4) * best_MulVal/(best_divisor * (best_MulVal + best_DivAddVal));
    
    /* reuse best_error to evaluate baud error*/
    if(baudrate>recalcbaud) best_error = baudrate - recalcbaud;
    else best_error = recalcbaud -baudrate;
    
    best_error = best_error * 100 / baudrate;
    
    if (best_error < UART_ACCEPTED_BAUDRATE_ERROR)
    {
        LPC_UART1->LCR = 0x83;//UART_LCR_DLAB_EN;
        LPC_UART1->DLM = UART_LOAD_DLM(best_divisor);
        LPC_UART1->DLL = UART_LOAD_DLL(best_divisor);
        /* Then reset DLAB bit */
        LPC_UART1->LCR = 0x03;//(~UART_LCR_DLAB_EN) & UART_LCR_BITMASK;
        LPC_UART1->FDR = ( UART_FDR_MULVAL(best_MulVal) | UART_FDR_DIVADDVAL(best_DivAddVal)) & UART_FDR_BITMASK;

        LPC_UART1->FCR = 0x07;                          // Enable and reset TX and RX FIFO
        LPC_UART1->TER |= UART_TER_TXEN;                // TX is enabled by default after a reset
        LPC_UART1->IER = IER_RBR;                       // Enable UART1 RX interrupt 

        errorStatus = SUCCESS;
    }
#endif
    
    // Test character
    //LPC_UART1->THR = '1';
        
    if(errorStatus ==SUCCESS)
        return (TRUE);
    else
        return (FALSE);

}



/******************************************************************************
 *  hex_to_ascii
 *
 *
 *
 ******************************************************************************/
char hex_to_ascii(char ch)
{
    if ( ch < 10) ch += 0x30;
    else ch += (0x41 - 0x0A);
    return (ch);
}


/******************************************************************************
 *  putChar
 *
 *
 *
 ******************************************************************************/
void putChar(unsigned char ch)
{  
    //while(!(LPC_UART1->LSR & LSR_THRE));  // Wait for Tx to complete
    //return(LPC_UART1->THR = ch);

    LPC_UART1->THR = ch;
}



/******************************************************************************
 *  writeString
 *
 *
 *
 ******************************************************************************/
void writeString (const char *p )    
{
 
 while( *p )
    {   
        // Read the UART status register and check for HOLDEMPTY bit.
        while( (LPC_UART1->LSR & LSR_THRE) == 0 )
        {
          ;
        }
        putChar( *p++ ); 
    }
}


/******************************************************************************
 *  getChar
 *
 *
 *
 ******************************************************************************/
unsigned char getChar (void)  
{                      
    return (LPC_UART1->RBR & UART_RBR_MASKBIT);    /* Read character from Serial Port */
}




